console.log('背景脚本已加载');
const UPDATE_CONFIG = {
    UPDATE_CHECK_URL: 'version.json'
};
async function checkForUpdates() {
    console.log('开始检查更新...');
    try {
        const manifest = chrome.runtime.getManifest();
        const currentVersion = manifest.version;
        console.log('当前版本:', currentVersion);
        console.log('正在从服务器获取更新信息...');
        const response = await fetch(UPDATE_CONFIG.UPDATE_CHECK_URL, {
            cache: 'no-cache',
            headers: {
                'Accept': 'application/json'
            }
        });
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        const data = await response.json();
        console.log('服务器返回数据:', data);
        if (data.version && data.version !== currentVersion) {
            console.log('发现新版本');
            const notificationId = 'update-' + Date.now();
            if (data.downloadUrl) {
                await chrome.storage.local.set({ 
                    ['updateUrl']: data.downloadUrl 
                });
            }
            const iconUrl = chrome.runtime.getURL('images/icon128.png');
            console.log('通知图标URL:', iconUrl);
            chrome.notifications.create(notificationId, {
                type: 'basic',
                iconUrl: iconUrl,
                title: '发现新版本',
                message: `发现新版本：${data.version}（当前版本：${currentVersion}）\n${data.description || ''}`,
                buttons: [
                    { title: '立即更新' },
                    { title: '稍后再说' }
                ],
                requireInteraction: true
            }, (notificationId) => {
                if (chrome.runtime.lastError) {
                    console.error('创建通知失败:', chrome.runtime.lastError);
                } else {
                    console.log('通知创建成功，ID:', notificationId);
                }
            });
            chrome.notifications.onButtonClicked.addListener(async function(clickedId, buttonIndex) {
                if (clickedId === notificationId) {
                    await chrome.notifications.clear(notificationId);
                    if (buttonIndex === 0) {
                        const result = await chrome.storage.local.get('updateUrl');
                        if (result.updateUrl) {
                            await chrome.tabs.create({ url: result.updateUrl });
                            await chrome.storage.local.remove('updateUrl');
                        }
                    }
                }
            });
            chrome.runtime.sendMessage({
                type: 'UPDATE_CHECK_RESULT',
                success: true,
                message: `发现新版本：${data.version} ${data.description || ''}`
            });
        } else {
            console.log('当前已是最新版本');
            chrome.runtime.sendMessage({
                type: 'UPDATE_CHECK_RESULT',
                success: true,
                message: '当前已是最新版本'
            });
        }
    } catch (error) {
        console.error('检查更新失败:', error);
        chrome.runtime.sendMessage({
            type: 'UPDATE_CHECK_RESULT',
            success: false,
            message: '检查更新失败：' + error.message
        });
    }
}
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    console.log('收到消息:', message);
    if (message.type === 'WEBPAGE_MESSAGE') {
        handleWebpageMessage(message.data, sender.tab)
            .catch(error => console.error('处理消息失败:', error));
        sendResponse({ received: true });
    } else if (message.type === 'CHECK_UPDATE') {
        console.log('收到检查更新请求');
        checkForUpdates()
            .catch(error => console.error('检查更新失败:', error));
        sendResponse({ received: true });
    }
    return false;
});
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete' && tab.url && tab.url.startsWith('http')) {
        chrome.tabs.get(tabId, (tabInfo) => {
            if (chrome.runtime.lastError) {
                console.log(`标签页 ${tabId} 不存在:`, chrome.runtime.lastError.message);
                return;
            }
            if (!tab.url.startsWith('chrome://') && !tab.url.startsWith('edge://')) {
                chrome.scripting.executeScript({
                    target: { tabId: tabId },
                    files: ['content-script.js']
                }).catch(err => {
                    if (err.message.includes('No tab with id')) {
                        console.log(`标签页 ${tabId} 已关闭或不可访问`);
                    } else {
                        console.error('注入内容脚本失败:', err);
                    }
                });
            }
        });
    }
});
async function handleWebpageMessage(request, tab) {
    console.log('处理网页消息:', request);
    let response = {
        type: 'EXTENSION_RESPONSE',
        action: request.action
    };
    try {
        if (!tab || !tab.id) {
            throw new Error('无效的标签页');
        }
        switch (request.action) {
            case 'ping':
                response.success = true;
                break;
            case 'getCookies':
                if (!request.domain) {
                    throw new Error('缺少域名参数');
                }
                const cookies = await getCookies(request.domain);
                response = {
                    ...response,
                    success: true,
                    cookies: cookies
                };
                break;
            case 'setCookies':
                if (!request.url || !request.cookieData) {
                    throw new Error('缺少必要参数');
                }
                const result = await setCookies(request.url, request.cookieData);
                response = {
                    ...response,
                    ...result
                };
                break;
            case 'setLocalStorage':
                if (!request.url || !request.storageData) {
                    throw new Error('缺少必要参数');
                }
                const storageResult = await setLocalStorage(request.url, request.storageData);
                response = {
                    ...response,
                    ...storageResult
                };
                break;
            default:
                throw new Error('未知的操作类型');
        }
    } catch (error) {
        console.error('处理消息时出错:', error);
        response.success = false;
        response.message = error.message;
    }
    try {
        if (tab && tab.id) {
            await chrome.tabs.sendMessage(tab.id, response);
        }
    } catch (error) {
        console.error('发送响应失败:', error);
        if (!error.message.includes('No tab with id')) {
            throw error;
        }
    }
}
function getCookies(domain) {
    return new Promise((resolve, reject) => {
        const targetDomain = domain.startsWith('.') ? domain.slice(1) : domain;
        chrome.cookies.getAll({ domain: targetDomain }, cookies => {
            if (chrome.runtime.lastError) {
                reject(new Error('获取Cookie失败: ' + chrome.runtime.lastError.message));
                return;
            }
            if (!cookies || cookies.length === 0) {
                reject(new Error('未找到任何Cookie'));
                return;
            }
            const cookieString = cookies.map(cookie => 
                `${cookie.name}=${cookie.value}`
            ).join('; ');
            resolve(cookieString);
        });
    });
}
function setCookies(url, cookieStr) {
    return new Promise(async (resolve, reject) => {
        try {
            const urlObj = new URL(url);
            let domain = urlObj.hostname;
            const isSecure = urlObj.protocol === 'https:';
            const cookies = cookieStr.split(';').map(str => str.trim());
            let successCount = 0;
            let errors = [];
            const oldCookies = await new Promise(resolve => {
                chrome.cookies.getAll({ domain }, resolve);
            });
            for (const cookie of oldCookies) {
                const cookieUrl = (cookie.secure ? "https://" : "http://") + 
                    (cookie.domain.charAt(0) === '.' ? cookie.domain.slice(1) : cookie.domain) + 
                    cookie.path;
                await new Promise(resolve => {
                    chrome.cookies.remove({
                        url: cookieUrl,
                        name: cookie.name
                    }, resolve);
                });
            }
            for (const cookie of cookies) {
                const firstEqualIndex = cookie.indexOf('=');
                if (firstEqualIndex === -1) {
                    errors.push(`无效的Cookie格式(缺少等号): ${cookie}`);
                    continue;
                }
                
                const name = cookie.substring(0, firstEqualIndex).trim();
                const value = cookie.substring(firstEqualIndex + 1).trim();
                
                if (!name || !value) {
                    errors.push(`无效的Cookie格式(名称或值为空): ${cookie}`);
                    continue;
                }
                try {
                    const domainVariations = [
                        domain.replace(/^[a-zA-Z0-9-]+\./, ''),
                        domain.replace(/^\./, ''),     // 去掉域名前面的点
                        domain,
                        `.${domain}`,
                        `www.${domain}`,
                        `.www.${domain}`
                    ];
                    let success = false;
                    for (const domainVar of domainVariations) {
                        const cookieOptions = {
                            url: url,
                            name: name,
                            value: value,
                            domain: domainVar,
                            path: '/',
                            secure: isSecure
                        };
                        console.log(`尝试设置cookie: ${name}，使用domain: ${domainVar}`, cookieOptions);
                        const result = await new Promise(resolve => {
                            chrome.cookies.set(cookieOptions, (cookie) => {
                                if (chrome.runtime.lastError) {
                                    console.log(`使用domain ${domainVar}设置失败:`, chrome.runtime.lastError);
                                    resolve(null);
                                } else {
                                    resolve(cookie);
                                }
                            });
                        });
                        if (result) {
                            success = true;
                            console.log(`成功设置cookie: ${name}，使用domain: ${domainVar}`);
                            break;
                        }
                    }
                    if (success) {
                        successCount++;
                    } else {
                        errors.push(`设置Cookie失败 (${name}): 所有domain变体都失败了`);
                    }
                } catch (error) {
                    console.error(`设置cookie出错: ${name}`, error);
                    errors.push(`设置Cookie失败 (${name}): ${error.message}`);
                }
            }
            resolve({
                success: successCount > 0,
                message: `成功设置 ${successCount}/${cookies.length} 个Cookie` +
                    (errors.length > 0 ? `\n错误信息:\n${errors.join('\n')}` : '')
            });
        } catch (error) {
            console.error('处理Cookie时出错:', error);
            reject(new Error('处理Cookie时出错: ' + error.message));
        }
    });
}
chrome.runtime.onInstalled.addListener((details) => {
    console.log('扩展已安装或更新:', details.reason);
});
chrome.runtime.onConnect.addListener(function(port) {
    console.log('建立了新的连接');
}); 
function setLocalStorage(url, storageData) {
    return new Promise(async (resolve, reject) => {
        try {
            if (!url || !storageData) {
                throw new Error('URL或LocalStorage数据不能为空');
            }
            const storageItems = storageData.split(';').filter(item => item.trim() !== '');
            if (storageItems.length === 0) {
                throw new Error('LocalStorage数据格式不正确');
            }
            const urlObj = new URL(url);
            const domain = urlObj.hostname;
            await chrome.storage.local.set({
                [`localStorage_${domain}`]: {
                    items: storageItems,
                    timestamp: Date.now()
                }
            });
            resolve({
                success: true,
                message: `LocalStorage数据已准备，将在页面加载时设置 ${storageItems.length} 个项目`,
                pendingItems: storageItems.length
            });
        } catch (error) {
            console.error('准备LocalStorage数据时出错:', error);
            reject(new Error('准备LocalStorage数据时出错: ' + error.message));
        }
    });
}
chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
    if (changeInfo.status === 'complete' && tab.url) {
        try {
            const url = new URL(tab.url);
            const domain = url.hostname;
            const storedData = await chrome.storage.local.get(`localStorage_${domain}`);
            const localStorageData = storedData[`localStorage_${domain}`];
            if (localStorageData && localStorageData.items && localStorageData.items.length > 0) {
                const now = Date.now();
                if (now - localStorageData.timestamp < 600000) {
                    const injectionResults = await chrome.scripting.executeScript({
                        target: { tabId },
                        func: (items) => {
                            const results = { 
                                success: [], 
                                error: [],
                                needsReload: false
                            };
                            let hasKeyDifferences = false;
                            items.forEach(item => {
                                try {
                                    const [key, ...valueParts] = item.split('=');
                                    const value = valueParts.join('=');
                                    if (!key || !value) {
                                        results.error.push(`无效的键值对: ${item}`);
                                        return;
                                    }
                                    const trimmedKey = key.trim();
                                    const trimmedValue = value.trim();
                                    const currentValue = localStorage.getItem(trimmedKey);
                                    if (currentValue !== trimmedValue) {
                                        hasKeyDifferences = true;
                                        localStorage.setItem(trimmedKey, trimmedValue);
                                        results.success.push(trimmedKey);
                                    }
                                } catch (error) {
                                    results.error.push(`设置 ${item} 时出错: ${error.message}`);
                                }
                            });
                            results.needsReload = hasKeyDifferences && results.success.length > 0;
                            try {
                                const hasAuthElements = 
                                    document.querySelector('.logout') || 
                                    document.querySelector('.user-info') || 
                                    document.querySelector('.avatar') ||
                                    document.querySelector('[aria-label="登出"]') ||
                                    document.getElementById('logout');
                                if (results.success.length > 0 && !hasAuthElements) {
                                    results.needsReload = true;
                                }
                            } catch (e) {
                            }
                            return results;
                        },
                        args: [localStorageData.items]
                    });
                    const results = injectionResults[0].result;
                    console.log(`已为 ${domain} 设置 ${results.success.length} 个LocalStorage项`);
                    await chrome.storage.local.remove(`localStorage_${domain}`);
                    if (results.needsReload) {
                        setTimeout(async () => {
                            await chrome.tabs.reload(tabId);
                            console.log(`${domain} 检测到需要刷新，已自动刷新页面`);
                        }, 500);
                    } else {
                        console.log(`${domain} 无需刷新，LocalStorage已成功应用`);
                    }
                } else {
                    await chrome.storage.local.remove(`localStorage_${domain}`);
                }
            }
        } catch (error) {
            console.error('设置LocalStorage时出错:', error);
        }
    }
}); 