document.addEventListener('DOMContentLoaded', function() {
    const manifest = chrome.runtime.getManifest();
    const version = manifest.version;
    const info = document.querySelector('.info');
    const versionInfo = document.createElement('p');
    versionInfo.textContent = `当前版本：${version}`;
    info.appendChild(versionInfo);
    const statusDiv = document.createElement('div');
    statusDiv.className = 'status';
    statusDiv.style.display = 'none';
    info.appendChild(statusDiv);
    document.getElementById('checkUpdate').addEventListener('click', function() {
        const button = this;
        button.textContent = '正在检查更新...';
        button.disabled = true;
        statusDiv.style.display = 'none';
        chrome.runtime.sendMessage({ type: 'CHECK_UPDATE' }, () => {
            chrome.runtime.onMessage.addListener(function listener(response) {
                if (response.type === 'UPDATE_CHECK_RESULT') {
                    statusDiv.textContent = response.message;
                    statusDiv.style.display = 'block';
                    statusDiv.style.backgroundColor = response.success ? '#e8f5e9' : '#ffebee';
                    button.textContent = '检查更新';
                    button.disabled = false;
                    chrome.runtime.onMessage.removeListener(listener);
                }
            });
            setTimeout(() => {
                if (button.disabled) {
                    statusDiv.textContent = '检查更新超时，请重试';
                    statusDiv.style.display = 'block';
                    statusDiv.style.backgroundColor = '#ffebee';
                    button.textContent = '检查更新';
                    button.disabled = false;
                }
            }, 30000);
        });
    });
    
    // 添加退出账号功能
    document.getElementById('logout').addEventListener('click', function() {
        const button = this;
        button.textContent = '正在清除数据...';
        button.disabled = true;
        statusDiv.style.display = 'none';
        
        chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
            if (tabs && tabs[0]) {
                const currentTab = tabs[0];
                try {
                    const currentUrl = new URL(currentTab.url);
                    const domain = currentUrl.hostname;
                    
                    // 处理各种可能的域名格式 (包括子域名)
                    const domainVariations = [
                        domain,
                        `.${domain}`,
                        domain.replace(/^www\./, ''),
                        `.${domain.replace(/^www\./, '')}`
                    ];
                    
                    // 计数器跟踪清除的cookie数量
                    let cookieCount = 0;
                    let processedVariations = 0;
                    
                    // 为每个域名变体清除cookie
                    domainVariations.forEach(domainVar => {
                        chrome.cookies.getAll({domain: domainVar}, function(cookies) {
                            if (cookies && cookies.length > 0) {
                                cookieCount += cookies.length;
                                cookies.forEach(cookie => {
                                    // 构建cookie的URL（包括协议、域名和路径）
                                    const cookieUrl = (cookie.secure ? "https://" : "http://") + 
                                        (cookie.domain.charAt(0) === '.' ? cookie.domain.slice(1) : cookie.domain) + 
                                        cookie.path;
                                    
                                    // 移除cookie
                                    chrome.cookies.remove({
                                        url: cookieUrl,
                                        name: cookie.name
                                    });
                                });
                            }
                            
                            // 检查是否已处理所有变体
                            processedVariations++;
                            if (processedVariations === domainVariations.length) {
                                // 所有cookie处理完毕，现在清除localStorage和sessionStorage
                                clearLocalStorageAndReload(currentTab, button, statusDiv, cookieCount);
                            }
                        });
                    });
                } catch (error) {
                    console.error('处理URL时出错:', error);
                    button.textContent = '退出账号';
                    button.disabled = false;
                    statusDiv.textContent = '退出账号失败: ' + error.message;
                    statusDiv.style.display = 'block';
                    statusDiv.style.backgroundColor = '#ffebee';
                }
            } else {
                button.textContent = '退出账号';
                button.disabled = false;
                statusDiv.textContent = '无法操作当前页面';
                statusDiv.style.display = 'block';
                statusDiv.style.backgroundColor = '#ffebee';
            }
        });
    });
    
    // 辅助函数：清除localStorage和sessionStorage并重新加载页面
    function clearLocalStorageAndReload(tab, button, statusDiv, cookieCount) {
        chrome.scripting.executeScript({
            target: {tabId: tab.id},
            func: () => {
                // 清除localStorage的多种方法
                try {
                    // 方法1：直接使用clear
                    localStorage.clear();
                    
                    // 方法2：逐个删除每个键
                    for (let i = 0; i < localStorage.length; i++) {
                        const key = localStorage.key(i);
                        localStorage.removeItem(key);
                    }
                    
                    // 方法3：遍历所有键删除
                    const keys = Object.keys(localStorage);
                    keys.forEach(key => localStorage.removeItem(key));
                    
                    // 清除sessionStorage
                    sessionStorage.clear();
                    
                    // 清除indexedDB
                    if (window.indexedDB) {
                        // 获取所有indexedDB数据库名称并删除
                        const databases = indexedDB.databases ? indexedDB.databases() : [];
                        if (databases && databases.then) {
                            databases.then(dbs => {
                                dbs.forEach(db => {
                                    if (db.name) indexedDB.deleteDatabase(db.name);
                                });
                            });
                        }
                    }
                    
                    return {
                        localStorage: localStorage.length === 0,
                        sessionStorage: sessionStorage.length === 0
                    };
                } catch (e) {
                    return { error: e.message };
                }
            }
        }).then(injectionResults => {
            const result = injectionResults[0].result;
            
            if (result.error) {
                button.textContent = '退出账号';
                button.disabled = false;
                statusDiv.textContent = `清除数据出错: ${result.error}`;
                statusDiv.style.display = 'block';
                statusDiv.style.backgroundColor = '#ffebee';
                return;
            }
            
            // 刷新当前页面
            chrome.tabs.reload(tab.id, function() {
                // 显示成功信息
                button.textContent = '退出账号';
                button.disabled = false;
                const storageStatus = result.localStorage ? '已清除' : '清除失败';
                statusDiv.textContent = `已退出当前网站账号，页面已刷新`;
                statusDiv.style.display = 'block';
                statusDiv.style.backgroundColor = '#e8f5e9';
            });
        }).catch(error => {
            console.error('清除数据失败:', error);
            button.textContent = '退出账号';
            button.disabled = false;
            statusDiv.textContent = '退出账号失败: ' + error.message;
            statusDiv.style.display = 'block';
            statusDiv.style.backgroundColor = '#ffebee';
        });
    }
}); 